<?php
/**
 * HRM Pagination Helper
 * Provides consistent pagination across all HRMS modules
 *
 * @package HR_Management
 * @since 2.1.0
 */

defined( 'ABSPATH' ) || exit;

class HRM_Pagination {
    
    /**
     * Default items per page
     */
    const DEFAULT_PER_PAGE = 10;
    
    /**
     * Available page size options
     */
    const PAGE_SIZE_OPTIONS = [10, 20, 25, 50, 100];
    
    /**
     * Get current page number from request
     *
     * @param string $param_name The query parameter name (default: 'paged')
     * @return int Current page number
     */
    public static function get_current_page( $param_name = 'paged' ) {
        $page = isset( $_GET[$param_name] ) ? absint( $_GET[$param_name] ) : 1;
        return max( 1, $page );
    }
    
    /**
     * Get items per page from request
     *
     * @param string $param_name The query parameter name (default: 'per_page')
     * @return int Items per page
     */
    public static function get_per_page( $param_name = 'per_page' ) {
        $per_page = isset( $_GET[$param_name] ) ? absint( $_GET[$param_name] ) : self::DEFAULT_PER_PAGE;
        
        // Ensure it's a valid option
        if ( ! in_array( $per_page, self::PAGE_SIZE_OPTIONS ) ) {
            $per_page = self::DEFAULT_PER_PAGE;
        }
        
        return $per_page;
    }
    
    /**
     * Calculate offset for database queries
     *
     * @param int $current_page Current page number
     * @param int $per_page Items per page
     * @return int Offset value
     */
    public static function get_offset( $current_page, $per_page ) {
        return ( $current_page - 1 ) * $per_page;
    }
    
    /**
     * Calculate total pages
     *
     * @param int $total_items Total number of items
     * @param int $per_page Items per page
     * @return int Total number of pages
     */
    public static function get_total_pages( $total_items, $per_page ) {
        return max( 1, ceil( $total_items / $per_page ) );
    }
    
    /**
     * Get pagination data array
     *
     * @param int $total_items Total number of items
     * @param int $current_page Current page number
     * @param int $per_page Items per page
     * @return array Pagination data
     */
    public static function get_pagination_data( $total_items, $current_page, $per_page ) {
        $total_pages = self::get_total_pages( $total_items, $per_page );
        $current_page = min( $current_page, $total_pages ); // Ensure current page doesn't exceed total
        
        $start_item = ( $current_page - 1 ) * $per_page + 1;
        $end_item = min( $current_page * $per_page, $total_items );
        
        return [
            'total_items' => $total_items,
            'per_page' => $per_page,
            'current_page' => $current_page,
            'total_pages' => $total_pages,
            'offset' => self::get_offset( $current_page, $per_page ),
            'start_item' => $start_item,
            'end_item' => $end_item,
            'has_previous' => $current_page > 1,
            'has_next' => $current_page < $total_pages
        ];
    }
    
    /**
     * Render complete pagination bar with page size selector and info
     *
     * @param array $args {
     *     @type int    $total_items   Total number of items
     *     @type int    $current_page  Current page number
     *     @type int    $per_page      Items per page
     *     @type string $base_url      Base URL for pagination links (optional for AJAX)
     *     @type array  $query_args    Additional query arguments to preserve
     *     @type string $context       Context identifier for JS (admin/frontend)
     *     @type bool   $show_sizes    Show page size selector
     *     @type bool   $ajax          Use AJAX pagination
     *     @type string $ajax_action   AJAX action name
     *     @type string $container_id  Container element ID to update
     * }
     */
    public static function render( $args = [] ) {
        $defaults = [
            'total_items' => 0,
            'current_page' => 1,
            'per_page' => self::DEFAULT_PER_PAGE,
            'base_url' => '',
            'query_args' => [],
            'context' => 'admin',
            'show_sizes' => true,
            'ajax' => false,
            'ajax_action' => '',
            'container_id' => '',
        ];
        
        $args = wp_parse_args( $args, $defaults );
        $pagination = self::get_pagination_data( $args['total_items'], $args['current_page'], $args['per_page'] );
        
        // Don't render if no items
        if ( $pagination['total_items'] == 0 ) {
            return;
        }
        
        $unique_id = uniqid( 'hrm_pg_' );
        ?>
        <div class="hrm-pagination-wrapper" id="<?php echo esc_attr( $unique_id ); ?>" 
             data-total-items="<?php echo esc_attr( $pagination['total_items'] ); ?>"
             data-current-page="<?php echo esc_attr( $pagination['current_page'] ); ?>"
             data-per-page="<?php echo esc_attr( $pagination['per_page'] ); ?>"
             data-total-pages="<?php echo esc_attr( $pagination['total_pages'] ); ?>"
             data-ajax="<?php echo $args['ajax'] ? 'true' : 'false'; ?>"
             data-ajax-action="<?php echo esc_attr( $args['ajax_action'] ); ?>"
             data-container="<?php echo esc_attr( $args['container_id'] ); ?>">
            
            <div class="hrm-pagination-row">
                <!-- Page Size Selector -->
                <?php if ( $args['show_sizes'] ) : ?>
                <div class="hrm-page-size-wrapper">
                    <label for="<?php echo esc_attr( $unique_id ); ?>_size">Show:</label>
                    <select id="<?php echo esc_attr( $unique_id ); ?>_size" 
                            class="hrm-page-size-select" 
                            onchange="hrmChangePagination(this, 'size')">
                        <?php foreach ( self::PAGE_SIZE_OPTIONS as $size ) : ?>
                            <option value="<?php echo esc_attr( $size ); ?>" <?php selected( $pagination['per_page'], $size ); ?>>
                                <?php echo esc_html( $size ); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <span class="hrm-page-size-label">per page</span>
                </div>
                <?php endif; ?>
                
                <!-- Pagination Info -->
                <div class="hrm-pagination-info">
                    <span class="hrm-pagination-showing">
                        <?php if ( $pagination['total_items'] > 0 ) : ?>
                            Showing <strong><?php echo esc_html( $pagination['start_item'] ); ?></strong> - 
                            <strong><?php echo esc_html( $pagination['end_item'] ); ?></strong> of 
                            <strong><?php echo esc_html( $pagination['total_items'] ); ?></strong> items
                        <?php else : ?>
                            No items found
                        <?php endif; ?>
                    </span>
                </div>
                
                <!-- Pagination Controls -->
                <?php if ( $pagination['total_pages'] > 1 ) : ?>
                <div class="hrm-pagination-controls">
                    <!-- Previous Button -->
                    <button type="button" 
                            class="hrm-pagination-btn hrm-pg-prev" 
                            <?php echo ! $pagination['has_previous'] ? 'disabled' : ''; ?>
                            onclick="hrmChangePagination(this, 'prev')"
                            data-page="<?php echo esc_attr( max( 1, $pagination['current_page'] - 1 ) ); ?>">
                        <span class="dashicons dashicons-arrow-left-alt2"></span>
                        <span class="btn-text">Previous</span>
                    </button>
                    
                    <!-- Page Numbers -->
                    <div class="hrm-page-numbers">
                        <?php echo self::render_page_numbers( $pagination['current_page'], $pagination['total_pages'] ); ?>
                    </div>
                    
                    <!-- Next Button -->
                    <button type="button" 
                            class="hrm-pagination-btn hrm-pg-next" 
                            <?php echo ! $pagination['has_next'] ? 'disabled' : ''; ?>
                            onclick="hrmChangePagination(this, 'next')"
                            data-page="<?php echo esc_attr( min( $pagination['total_pages'], $pagination['current_page'] + 1 ) ); ?>">
                        <span class="btn-text">Next</span>
                        <span class="dashicons dashicons-arrow-right-alt2"></span>
                    </button>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render page number buttons
     *
     * @param int $current_page Current page number
     * @param int $total_pages Total number of pages
     * @return string HTML output
     */
    public static function render_page_numbers( $current_page, $total_pages ) {
        $output = '';
        $range = 2; // Pages to show on each side of current
        
        for ( $i = 1; $i <= $total_pages; $i++ ) {
            if ( 
                $i == 1 || 
                $i == $total_pages || 
                ( $i >= $current_page - $range && $i <= $current_page + $range ) 
            ) {
                $active_class = ( $i == $current_page ) ? ' active' : '';
                $output .= sprintf(
                    '<button type="button" class="hrm-pagination-btn hrm-pg-num%s" onclick="hrmChangePagination(this, \'page\')" data-page="%d">%d</button>',
                    $active_class,
                    $i,
                    $i
                );
            } elseif ( 
                $i == $current_page - $range - 1 || 
                $i == $current_page + $range + 1 
            ) {
                $output .= '<span class="hrm-pagination-ellipsis">...</span>';
            }
        }
        
        return $output;
    }
    
    /**
     * Render simple pagination for frontend cards/lists
     *
     * @param array $args Same as render() method
     */
    public static function render_simple( $args = [] ) {
        $defaults = [
            'total_items' => 0,
            'current_page' => 1,
            'per_page' => self::DEFAULT_PER_PAGE,
            'show_sizes' => false,
        ];
        
        $args = wp_parse_args( $args, $defaults );
        $args['show_sizes'] = false; // Force no size selector for simple
        
        self::render( $args );
    }
    
    /**
     * Render pagination for admin tables with links (server-side)
     *
     * @param array $args {
     *     @type int    $total_items  Total number of items
     *     @type int    $current_page Current page number
     *     @type int    $per_page     Items per page
     *     @type string $base_url     Base URL for pagination links
     *     @type array  $query_args   Additional query arguments to preserve
     * }
     */
    public static function render_links( $args = [] ) {
        $defaults = [
            'total_items' => 0,
            'current_page' => 1,
            'per_page' => self::DEFAULT_PER_PAGE,
            'base_url' => '',
            'query_args' => [],
        ];
        
        $args = wp_parse_args( $args, $defaults );
        $pagination = self::get_pagination_data( $args['total_items'], $args['current_page'], $args['per_page'] );
        
        if ( $pagination['total_pages'] <= 1 ) {
            return;
        }
        
        // Build base URL with preserved query args
        $base_url = $args['base_url'] ?: remove_query_arg( 'paged' );
        if ( ! empty( $args['query_args'] ) ) {
            $base_url = add_query_arg( $args['query_args'], $base_url );
        }
        
        ?>
        <div class="hrm-pagination">
            <div class="hrm-pagination-info">
                Showing <?php echo esc_html( $pagination['start_item'] ); ?> - 
                <?php echo esc_html( $pagination['end_item'] ); ?> of 
                <?php echo esc_html( $pagination['total_items'] ); ?> items
            </div>
            <div class="hrm-pagination-controls">
                <?php if ( $pagination['has_previous'] ) : ?>
                    <a href="<?php echo esc_url( add_query_arg( 'paged', $pagination['current_page'] - 1, $base_url ) ); ?>" 
                       class="hrm-pagination-btn">
                        <span class="dashicons dashicons-arrow-left-alt2"></span> Previous
                    </a>
                <?php endif; ?>
                
                <?php
                $range = 2;
                for ( $i = 1; $i <= $pagination['total_pages']; $i++ ) :
                    if ( 
                        $i == 1 || 
                        $i == $pagination['total_pages'] || 
                        ( $i >= $pagination['current_page'] - $range && $i <= $pagination['current_page'] + $range ) 
                    ) :
                        $active_class = ( $i == $pagination['current_page'] ) ? ' active' : '';
                        ?>
                        <a href="<?php echo esc_url( add_query_arg( 'paged', $i, $base_url ) ); ?>" 
                           class="hrm-pagination-btn<?php echo esc_attr( $active_class ); ?>">
                            <?php echo esc_html( $i ); ?>
                        </a>
                        <?php
                    elseif ( 
                        $i == $pagination['current_page'] - $range - 1 || 
                        $i == $pagination['current_page'] + $range + 1 
                    ) :
                        ?>
                        <span class="hrm-pagination-ellipsis">...</span>
                        <?php
                    endif;
                endfor;
                ?>
                
                <?php if ( $pagination['has_next'] ) : ?>
                    <a href="<?php echo esc_url( add_query_arg( 'paged', $pagination['current_page'] + 1, $base_url ) ); ?>" 
                       class="hrm-pagination-btn">
                        Next <span class="dashicons dashicons-arrow-right-alt2"></span>
                    </a>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render JavaScript for pagination handling
     * Should be called once per page
     */
    public static function render_script() {
        static $rendered = false;
        
        if ( $rendered ) {
            return;
        }
        $rendered = true;
        
        ?>
        <script>
        /**
         * Global pagination handler
         */
        function hrmChangePagination(element, action) {
            var wrapper = jQuery(element).closest('.hrm-pagination-wrapper');
            var currentPage = parseInt(wrapper.data('current-page'));
            var totalPages = parseInt(wrapper.data('total-pages'));
            var perPage = parseInt(wrapper.data('per-page'));
            var isAjax = wrapper.data('ajax') === true || wrapper.data('ajax') === 'true';
            var ajaxAction = wrapper.data('ajax-action');
            var containerId = wrapper.data('container');
            
            var newPage = currentPage;
            var newPerPage = perPage;
            
            switch(action) {
                case 'prev':
                    newPage = Math.max(1, currentPage - 1);
                    break;
                case 'next':
                    newPage = Math.min(totalPages, currentPage + 1);
                    break;
                case 'page':
                    newPage = parseInt(jQuery(element).data('page'));
                    break;
                case 'size':
                    newPerPage = parseInt(jQuery(element).val());
                    newPage = 1; // Reset to first page when changing size
                    break;
            }
            
            if (isAjax && ajaxAction && containerId) {
                // AJAX pagination
                hrmAjaxPaginate(wrapper, newPage, newPerPage, ajaxAction, containerId);
            } else {
                // Server-side pagination (reload with URL params)
                var url = new URL(window.location.href);
                url.searchParams.set('paged', newPage);
                url.searchParams.set('per_page', newPerPage);
                window.location.href = url.toString();
            }
        }
        
        /**
         * AJAX pagination handler
         */
        function hrmAjaxPaginate(wrapper, page, perPage, action, containerId) {
            var container = jQuery('#' + containerId);
            if (!container.length) return;
            
            // Show loading state
            container.css('opacity', '0.5').css('pointer-events', 'none');
            
            var ajaxData = {
                action: action,
                nonce: typeof hrm_ajax !== 'undefined' ? hrm_ajax.nonce : (typeof hrm_admin_params !== 'undefined' ? hrm_admin_params.nonce : ''),
                paged: page,
                per_page: perPage
            };
            
            // Collect current filter values if any
            container.closest('form').find('input, select').each(function() {
                var name = jQuery(this).attr('name');
                if (name && name !== 'paged' && name !== 'per_page') {
                    ajaxData[name] = jQuery(this).val();
                }
            });
            
            jQuery.ajax({
                url: typeof ajaxurl !== 'undefined' ? ajaxurl : (typeof hrm_ajax !== 'undefined' ? hrm_ajax.ajaxurl : '/wp-admin/admin-ajax.php'),
                type: 'POST',
                data: ajaxData,
                success: function(response) {
                    if (response.success && response.data.html) {
                        container.html(response.data.html);
                    }
                    container.css('opacity', '1').css('pointer-events', 'auto');
                },
                error: function() {
                    container.css('opacity', '1').css('pointer-events', 'auto');
                    console.error('Pagination AJAX request failed');
                }
            });
        }
        
        /**
         * Client-side pagination for existing elements
         * Used for Leave History cards and similar client-paginated content
         */
        function hrmClientPaginate(containerId, itemSelector, page, perPage) {
            var container = jQuery('#' + containerId);
            var items = container.find(itemSelector);
            var totalItems = items.length;
            var totalPages = Math.ceil(totalItems / perPage);
            
            // Hide all items
            items.hide();
            
            // Show items for current page
            var startIndex = (page - 1) * perPage;
            var endIndex = Math.min(startIndex + perPage, totalItems);
            
            items.slice(startIndex, endIndex).show();
            
            // Update pagination info
            var paginationWrapper = container.siblings('.hrm-pagination-wrapper');
            paginationWrapper.find('.hrm-pagination-showing').html(
                'Showing <strong>' + (startIndex + 1) + '</strong> - <strong>' + endIndex + '</strong> of <strong>' + totalItems + '</strong> items'
            );
            
            // Update button states
            paginationWrapper.find('.hrm-pg-prev').prop('disabled', page <= 1).data('page', Math.max(1, page - 1));
            paginationWrapper.find('.hrm-pg-next').prop('disabled', page >= totalPages).data('page', Math.min(totalPages, page + 1));
            
            // Update active page
            paginationWrapper.find('.hrm-pg-num').removeClass('active');
            paginationWrapper.find('.hrm-pg-num[data-page="' + page + '"]').addClass('active');
            
            // Update wrapper data
            paginationWrapper.data('current-page', page);
            
            return { page: page, totalPages: totalPages };
        }
        </script>
        <?php
    }
}
